// Copyright 2025 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package mldsa

import (
	"bytes"
	"crypto/internal/fips140"
	_ "crypto/internal/fips140/check"
	"crypto/internal/fips140/sha256"
	"errors"
	"sync"
)

func fipsPCT(priv *PrivateKey) {
	fips140.PCT("ML-DSA sign and verify PCT", func() error {
		μ := make([]byte, 64)
		sig, err := SignExternalMuDeterministic(priv, μ)
		if err != nil {
			return err
		}
		return VerifyExternalMu(priv.PublicKey(), μ, sig)
	})
}

var fipsSelfTest = sync.OnceFunc(func() {
	fips140.CAST("ML-DSA-44", fips140CAST)
})

// fips140CAST covers all rejection sampling paths, as recommended by IG 10.3.A,
// and as tested by TestCASTRejectionPaths. It tests only one parameter set as
// allowed by Note26. It tests the modified version of Algorithm 7 and 8 with a
// fixed mu/μ, as allowed by IG 10.3.A, Resolution 15. It compares sk and not
// pk, because H(pk) is part of sk, as allowed by the same Resolution. It
// compares the results with hashes instead of values, to avoid embedding several
// kilobytes of test vectors in every binary, as allowed by GeneralNote7.
func fips140CAST() error {
	// From https://pages.nist.gov/ACVP/draft-celi-acvp-ml-dsa.html#table-1.
	var seed = &[32]byte{
		0x5c, 0x62, 0x4f, 0xcc, 0x18, 0x62, 0x45, 0x24,
		0x52, 0xd0, 0xc6, 0x65, 0x84, 0x0d, 0x82, 0x37,
		0xf4, 0x31, 0x08, 0xe5, 0x49, 0x9e, 0xdc, 0xdc,
		0x10, 0x8f, 0xbc, 0x49, 0xd5, 0x96, 0xe4, 0xb7,
	}
	var μ = &[64]byte{
		0x2a, 0xd1, 0xc7, 0x2b, 0xb0, 0xfc, 0xbe, 0x28,
		0x09, 0x9c, 0xe8, 0xbd, 0x2e, 0xd8, 0x36, 0xdf,
		0xeb, 0xe5, 0x20, 0xaa, 0xd3, 0x8f, 0xba, 0xc6,
		0x6e, 0xf7, 0x85, 0xa3, 0xcf, 0xb1, 0x0f, 0xb4,
		0x19, 0x32, 0x7f, 0xa5, 0x78, 0x18, 0xee, 0x4e,
		0x37, 0x18, 0xda, 0x4b, 0xe4, 0x8d, 0x24, 0xb5,
		0x9a, 0x20, 0x8f, 0x88, 0x07, 0x27, 0x1f, 0xdb,
		0x7e, 0xda, 0x6e, 0x60, 0x14, 0x1b, 0xd2, 0x63,
	}
	var skHash = []byte{
		0x29, 0x37, 0x49, 0x51, 0xcb, 0x2b, 0xc3, 0xcd,
		0xa7, 0x31, 0x5c, 0xe7, 0xf0, 0xab, 0x99, 0xc7,
		0xd2, 0xd6, 0x52, 0x92, 0xe6, 0xc5, 0x15, 0x6e,
		0x8a, 0xa6, 0x2a, 0xc1, 0x4b, 0x14, 0x12, 0xaf,
	}
	var sigHash = []byte{
		0xdc, 0xc7, 0x1a, 0x42, 0x1b, 0xc6, 0xff, 0xaf,
		0xb7, 0xdf, 0x0c, 0x7f, 0x6d, 0x01, 0x8a, 0x19,
		0xad, 0xa1, 0x54, 0xd1, 0xe2, 0xee, 0x36, 0x0e,
		0xd5, 0x33, 0xce, 0xcd, 0x5d, 0xc9, 0x80, 0xad,
	}
	priv := newPrivateKey(seed, params44)
	H := sha256.New()
	H.Write(TestingOnlyPrivateKeySemiExpandedBytes(priv))
	if !bytes.Equal(H.Sum(nil), skHash) {
		return errors.New("unexpected private key hash")
	}
	var random [32]byte
	sig := signInternal(priv, μ, &random)
	H.Reset()
	H.Write(sig)
	if !bytes.Equal(H.Sum(nil), sigHash) {
		return errors.New("unexpected signature hash")
	}
	return verifyInternal(priv.PublicKey(), μ, sig)
}
